/*
 * Copyright (C) 2023 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.health.connect;

import android.annotation.NonNull;
import android.annotation.SuppressLint;
import android.health.connect.aidl.ReadRecordsRequestParcel;
import android.health.connect.datatypes.Metadata;
import android.health.connect.datatypes.Record;
import android.os.OutcomeReceiver;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Executor;

/**
 * A request class to represent request based on id filters with ids for {@link
 * HealthConnectManager#readRecords(ReadRecordsRequest, Executor, OutcomeReceiver)}
 *
 * @param <T> the type of the Record for the request
 */
public final class ReadRecordsRequestUsingIds<T extends Record> extends ReadRecordsRequest<T> {
    /** List of {@link RecordIdFilter} */
    private final List<RecordIdFilter> mRecordIdFiltersList;

    /**
     * @see Builder
     */
    private ReadRecordsRequestUsingIds(
            @NonNull Class<T> recordType, @NonNull List<RecordIdFilter> recordIdFiltersList) {
        super(recordType);
        Objects.requireNonNull(recordIdFiltersList);
        mRecordIdFiltersList = recordIdFiltersList;
    }

    /** Returns List of RecordId */
    @NonNull
    public List<RecordIdFilter> getRecordIdFilters() {
        return mRecordIdFiltersList;
    }

    /**
     * Returns an object of ReadRecordsRequestParcel to carry read request
     *
     * @hide
     */
    @NonNull
    public ReadRecordsRequestParcel toReadRecordsRequestParcel() {
        return new ReadRecordsRequestParcel(this);
    }

    /** Builder class for {@link ReadRecordsRequestUsingIds} */
    public static final class Builder<T extends Record> {
        private final Class<T> mRecordType;
        private final List<RecordIdFilter> mRecordIdFiltersList = new ArrayList<>();

        /**
         * @param recordType Record class for which the id is being set
         */
        public Builder(@NonNull Class<T> recordType) {
            Objects.requireNonNull(recordType);

            mRecordType = recordType;
        }

        /**
         * @param id Identifier generated by the platform and returned by {@link
         *     HealthConnectManager#insertRecords}
         */
        @NonNull
        @SuppressLint("MissingGetterMatchingBuilder")
        public Builder<T> addId(@NonNull String id) {
            mRecordIdFiltersList.add(RecordIdFilter.fromId(mRecordType, id));
            return this;
        }

        /**
         * @param clientRecordId identifier that was set while inserting the record
         * @see Metadata
         */
        @SuppressLint("MissingGetterMatchingBuilder")
        @NonNull
        public Builder<T> addClientRecordId(@NonNull String clientRecordId) {
            mRecordIdFiltersList.add(
                    RecordIdFilter.fromClientRecordId(mRecordType, clientRecordId));
            return this;
        }

        /** Returns Object of {@link ReadRecordsRequestUsingIds} */
        @NonNull
        public ReadRecordsRequestUsingIds<T> build() {
            if (mRecordIdFiltersList.isEmpty()) {
                throw new IllegalArgumentException(
                        "RecordIdFilter list is empty, "
                                + "Either record id or client record id must be set");
            }
            return new ReadRecordsRequestUsingIds<>(mRecordType, mRecordIdFiltersList);
        }
    }
}
